/*
  Base class for red and blue goals for Bombing Run.
  
  Author Mark Caldwell aka W@rHe@d of The Reliquary 
*/

class UTBRGoal extends UTGameObjective
	hidecategories(Objective,VoiceMessage,Announcements)
	abstract;

var()	float				MidFieldHighZOffset;
var()	float				MidFieldLowZOffset;
var()	float				NearBaseRadius;
var()	float				BaseExitTime;		// how long it takes to get entirely away from the base
 

var() StaticMeshComponent Mesh;
var() CylinderComponent	NavigationCylinderComponent;

/** Stops bot from lobbing ball away from this goal when it's the home goal */
var() bool DisableLobbingBallAway;

var Array<SoundNodeWave> NearLocationSpeech;
var Array<SoundNodeWave> MidfieldHighSpeech;
var Array<SoundNodeWave> MidfieldLowSpeech;


simulated function PostBeginPlay()
{
	Enable('Touch');

	Super(UTGameObjective).PostBeginPlay();

	if ( Role < ROLE_Authority )
		return;

	if ( UTBRGame(WorldInfo.Game) != None )
	{
		UTBRGame(WorldInfo.Game).RegisterGoal(self, DefenderTeamIndex);		
	}
}

event actor GetBestViewTarget()
{
	return self;
}

simulated function int GetLocationMessageIndex(UTBot B, Pawn StatusPawn)
{
	local float Dist, ZDiff;

	Dist = VSize(StatusPawn.Location - Location);
	if ( Dist < BaseRadius )
	{
		return 0;
	}
	if ( Dist < NearBaseRadius )
	{
		return 1;
	}

	// Midfield high or low?
	ZDiff = StatusPawn.Location.Z - Location.Z;
	if ( ZDiff > MidFieldHighZOffset )
	{
		return 2;
	}
	if ( ZDiff < MidFieldLowZOffset )
	{
		return 3;
	}

	return 10;
}

simulated function SoundNodeWave GetLocationSpeechFor(PlayerController PC, int LocationSpeechOffset, int MessageIndex)
{
	if ( MessageIndex == 0 )
	{
		// inside the base
		return (LocationSpeechOffset < LocationSpeech.Length) ? LocationSpeech[LocationSpeechOffset] : None;
	}
	else if ( MessageIndex == 1 )
	{
		// near the base
		return (LocationSpeechOffset < NearLocationSpeech.Length) ? NearLocationSpeech[LocationSpeechOffset] : None;
	}
	else if ( MessageIndex == 2 )
	{
		// Midfield High
		return (LocationSpeechOffset < MidFieldHighSpeech.Length) ? MidFieldHighSpeech[LocationSpeechOffset] : None;
	}
	else if ( MessageIndex == 3 )
	{
		// Midfield Low
		return (LocationSpeechOffset < MidFieldLowSpeech.Length) ? MidFieldLowSpeech[LocationSpeechOffset] : None;
	}

	return None;
}

function ObjectiveChanged()
{
}

event Bump( Actor Other, PrimitiveComponent OtherComp, Vector HitNormal )
{
	Disable('Bump');	
    GoalTouched(Other);	
	Enable('Bump');	    
}
	
event Touch( Actor Other, PrimitiveComponent OtherComp, vector HitLocation, vector HitNormal )
{
	Disable('Touch');	
    GoalTouched(Other);
  	Enable('Touch');
}

//called when goal is touched by anything, to see if a goal score should be registered
function GoalTouched(Actor other)
{
    local Pawn pawn;
    local bool isBallThrowScore;
    local UTBRBall ball;  
    local UTPlayerReplicationInfo pri;

    if (UTBRGame(WorldInfo.Game).Test_NoScore && (WorldInfo.Netmode == NM_Standalone))
    {
        return;
    }
       
    if (UTVehicle(other) != none)
    {
        other = UTVehicle(other).Driver;        
    } 
    
    pawn = Pawn(Other);
    
    if ((pawn != none) && (pawn.GetTeamNum() == DefenderTeamIndex))
    {
	    //goal kills anyone of same team touching it
	    pawn.Health = -1;
		pawn.Died( None, class'DmgType_Suicided', pawn.Location );
		return;
    }
     
    if (UTBRBall(other) != none)
    {
        if (bool(UTBRGame(WorldInfo.Game).Settings.DisableShootingScore) || (! UTBRBall(other).ShotByBallLauncher))
        {
	        return;
        }
        
        ball = UTBRBall(other);
        
        isBallThrowScore = true;
        
        pri = ball.LastHolderPRI;
        
	    if (pri == none)
	    {
		    return;
	    }                 
    }
    else
    {
	    if (pawn == none)
	    {
		    return;
	    }
  	       
	    pri = UTBRGame(WorldInfo.Game).GetPlayerReplicationInfo(pawn);
	    
	    if (pri == none)
	    {
		    return;
	    }
	     
	    ball = UTBRBall(pri.GetFlag());
	    
	    if (ball == none)
	    {
		    return;
	    }
	}
       

   	if (pri.GetTeamNum() == DefenderTeamIndex)
	{
		return;
	}
	
	if ((pri.GetTeamNum() < 0) || (pri.GetTeamNum() > 1))
	{
		return;
	}
	
	UTBRGame(WorldInfo.Game).ScoreBall(ball, isBallThrowScore);
	
      Spawn(class'UTBRGoalExplosion',,,location + mesh.translation);
	
	if ( !UTBRGame(WorldInfo.Game).bGameEnded )
	{
		UTBRGame(WorldInfo.Game).StartNewRound();	
	}

      ball.GoalScore();	
}

simulated event bool IsActive() 
{
	return true;
}


defaultproperties
{
   BaseExitTime=8.000000  
   DrawScale = 1.0
     
   //note: this actor does not use a collision Cylinder. Instead, it used mesh collision so that it can accurately detect
   //when ball or runner touches it for the score.
         
   Begin Object Class=StaticMeshComponent Name=StaticMeshComponent0 ObjName=StaticMeshComponent0 Archetype=StaticMeshComponent'Engine.Default__StaticMeshComponent'
      StaticMesh=StaticMesh'BombingRunGraphics.GoalDisc' 
      CullDistance=7000.000000     
      CachedCullDistance=7000.000000
      bUseAsOccluder=False
      CastShadow=False
      bForceDirectLightMap=True
      bCastDynamicShadow=False
      LightingChannels=(BSP=True,Static=True,Dynamic=True)
      CollideActors=True  
      Translation=(X=0.000000,Y=0.000000,Z=0.000000)
      Name="StaticMeshComponent0"
      ObjectArchetype=StaticMeshComponent'Engine.Default__StaticMeshComponent'
      BlockRigidBody=false          
   End Object

   BlockRigidBody=false 
   bHasSensor=True
   //IconCoords=(U=377.000000,V=438.000000,UL=45.000000,VL=44.000000)
   IconCoords=(U=537.000000,V=296.000000,UL=46.000000,VL=31.000000)   
 
   GoodSprite=None
   BadSprite=None

   Mesh=StaticMeshComponent0
          
   Components(0)=Arrow
   Components(1)=PathRenderer
   Components(2)=StaticMeshComponent0
   Components(3)=none
   Components(4)=none
     
   RemoteRole=ROLE_SimulatedProxy
   bStatic=False
   bAlwaysRelevant=True
	     	
   bCollideActors=True
   Name="Default__UTBRGoal"   
   ObjectArchetype=UTGameObjective'UTGame.Default__UTGameObjective'   
   
   bMovable=true   
   bHidden=True
   CollisionComponent=StaticMeshComponent0

   //needed to stop actor from dropping down in editor
   bNotBased=true
      
   //this had been needed to keep the goal from dropping down when rebuilding ai paths in map editor,
   //but now we use bNotBased to do this.
   //if bNotBased is false, editor uses NavigationPoint.CylinderComponent to try to drop a nav point flush with 
   //whatever base it can find below the point, and uses it to build navigation paths.
   //if it can't find a proper base then an error is generated.
   //if CylinderComponent is not present an error is generated.
   //Class=CylinderComponent
   Begin Object Name=CollisionCylinder ObjName=CollisionCylinder Archetype=CylinderComponent'Engine.Default__CylinderComponent'
      CollisionHeight=125
      CollisionRadius=125
      Name="CollisionCylinder"
      ObjectArchetype=CylinderComponent'Engine.Default__CylinderComponent'
      BlockRigidBody=false
   End Object   
   CylinderComponent=CollisionCylinder   
   NavigationCylinderComponent=CollisionCylinder
   
   //need this to make things show up on minimap
   MaxSensorRange=999999999999.0
}
